﻿using System;
using System.IO;
using System.Linq;
using System.Diagnostics;

namespace AHOBPR_Job_Runner
{
    public abstract class LoggedJob
    {
        private const String ArchivedLogFileFolder = "Archived Log Files";
        protected abstract String LogFileName { get; }
        public abstract String Id { get; }

        public abstract void Run();

        protected void StartLog()
        {
            Manage();
            LogMessageToFile("------------------------------ Build Version: " + GetBuildVersion() + "---------------------------------------");
        }
        
        protected void CheckDiskSpace()
        {
        //    if (GetFreeDiskSpace() < Convert.ToInt64(System.Configuration.ConfigurationSettings.AppSettings["FreeDiskSpaceLimitMB"]) * 1000000)
            if (GetFreeDiskSpace() < Convert.ToInt64(System.Configuration.ConfigurationManager.AppSettings["FreeDiskSpaceLimitMB"]) * 1000000)
            {
   //             LogMessageToFile("Warning: Total free disk space is less than " + System.Configuration.ConfigurationSettings.AppSettings["FreeDiskSpaceLimitMB"] + " megabytes!!!");
                LogMessageToFile("Warning: Total free disk space is less than " + System.Configuration.ConfigurationManager.AppSettings["FreeDiskSpaceLimitMB"] + " megabytes!!!");
            }
        }

        protected void LogErrorMessage(Exception ex)
        {
            using (StreamWriter writer = File.AppendText(LogFileName))
            {
                string message = string.Empty;
                StackTrace stack = new StackTrace(ex, true);

                writer.WriteLine("{0}: {1}", DateTime.Now.ToShortDateString() + " " + DateTime.Now.ToLongTimeString(), "EXCEPTION: " + ex.Message.Replace("\r\n", "\r\n                       "));
                foreach (StackFrame frame in stack.GetFrames())
                {
                    if (string.IsNullOrEmpty(frame.GetFileName()))
                    {
                        message = string.Format("METHOD: {0}", frame.GetMethod().ToString());
                    }
                    else
                    {
                        message = string.Format("FILE: {0}, LINE: {1}, METHOD: {2}",
                            frame.GetFileName(), frame.GetFileLineNumber().ToString(), frame.GetMethod().ToString());
                    }

                    writer.WriteLine("                       " + message);
                }
            }
        }

        protected void LogMessageToFile(string message)
        {
            using (StreamWriter writer = File.AppendText(LogFileName))
            {
                writer.WriteLine("{0}: {1}", DateTime.Now.ToShortDateString() + " " + DateTime.Now.ToLongTimeString(), message);
            }
        }

        /// <summary>
        /// If the log file is too large, it will have performance impact for the SQL Transporter. To solve this problem,
        /// we will rename the log file from AWARE_SQL_Transporter.log to AWARE_SQL_Transporter_yyyyMMddhhmmss.log when its size > set size (configurable value)
        /// 
        /// We also want to only keep the last x (configurable value) number of the log files to prevent the disk space full issue.
        /// </summary>
        private void Manage()
        {
            // rename the log file if it's size exceeds the max size allowed
            FileInfo file = new FileInfo(LogFileName);
            long logFileSize = 0;
            string archivedLogFileFolder = Directory.GetCurrentDirectory() + "\\" + ArchivedLogFileFolder;

            // if the log file does not exist yet, the file size is 0
            try
            {
                logFileSize = file.Length;
            }
            catch
            {
                logFileSize = 0;
            }
          //  if (logFileSize > Convert.ToInt64(System.Configuration.ConfigurationSettings.AppSettings["MaxSizePerLogFileMB"]) * 1000000)
            if (logFileSize > Convert.ToInt64(System.Configuration.ConfigurationManager.AppSettings["MaxSizePerLogFileMB"]) * 1000000)
            {
                if (!Directory.Exists(archivedLogFileFolder))
                {
                    Directory.CreateDirectory(archivedLogFileFolder);
                }
                File.Move(LogFileName, archivedLogFileFolder + "\\" + LogFileName.Replace(".log", "_" + DateTime.Now.ToString("yyyyMMddhhmmss") + ".log"));
            }

            // Delete the oldest log file if the number of the log files exceeds the maximum number of log files allowed.
            if (Directory.Exists(archivedLogFileFolder))
            {
         //       int numberOfLogFilesToKeep = Convert.ToInt32(System.Configuration.ConfigurationSettings.AppSettings["MaxNumberOfLogFilesToKeep"]);
                int numberOfLogFilesToKeep = Convert.ToInt32(System.Configuration.ConfigurationManager.AppSettings["MaxNumberOfLogFilesToKeep"]);
                var files = new DirectoryInfo(archivedLogFileFolder).EnumerateFiles()
                    .Where(f => f.FullName.EndsWith(".log"))
                    .OrderByDescending(f => f.LastWriteTime)
                    .Skip(numberOfLogFilesToKeep)
                    .ToList();

                files.ForEach(f => f.Delete());
            }
        }

        private static long GetFreeDiskSpace()
        {
            return new DriveInfo(Path.GetPathRoot(System.Diagnostics.Process.GetCurrentProcess().MainModule.FileName).Substring(0, 1)).AvailableFreeSpace;
        }

        private static string GetBuildVersion()
        {
         //   return System.Configuration.ConfigurationSettings.AppSettings["ApplicationVersion"];
            return System.Configuration.ConfigurationManager.AppSettings["ApplicationVersion"];
        }
    }
}
